﻿using System;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;

using System.Diagnostics;

using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;


namespace ZGF.Revit
{

    [Autodesk.Revit.Attributes.Transaction(Autodesk.Revit.Attributes.TransactionMode.Manual)]
    [Autodesk.Revit.Attributes.Regeneration(Autodesk.Revit.Attributes.RegenerationOption.Manual)]
    public class SyncIntElevViewNamesWithRooms : IExternalCommand
    {
        public Result Execute(Autodesk.Revit.UI.ExternalCommandData commandData,
                                                ref string message,
                                                ElementSet elements)
        {

            DialogResult returnval = DialogResult.OK;

            try
            {
                
                ElevationViewsDialog elevRenamer = new ElevationViewsDialog(commandData);
                returnval = elevRenamer.ShowDialog();
                                
            }
            catch
            {
                returnval = DialogResult.Abort;
            }

            switch (returnval)
            {
                case DialogResult.OK:
                    return Result.Succeeded;

                case DialogResult.Abort:
                    return Result.Failed;

                default:
                    return Result.Cancelled;

            }
        }
    }


    /// <summary>
    /// Test class for elevation view dialog
    /// </summary>
    class ElevationViews
    {
        // M E M B E R S
        Document m_doc;
        List<Autodesk.Revit.DB.ViewSection> m_int_elevation_views = new List<Autodesk.Revit.DB.ViewSection>();

        double m_north_angle_offset;

        // C O N S T R U C T O R
        public ElevationViews(ExternalCommandData commandData, ElementSet theElements)
        {
            
            m_doc = commandData.Application.ActiveUIDocument.Document;

            Autodesk.Revit.UI.Selection.Selection es = commandData.Application.ActiveUIDocument.Selection;


            // If project has a rotated True North, this will adjust
            // the orientation portion of the view name.
            ProjectLocationSet pls = m_doc.ProjectLocations;
            foreach (ProjectLocation p in pls)
            {
                if (p.Name == "Internal")
                {
                    ProjectPosition pos = p.get_ProjectPosition(XYZ.Zero);
                    m_north_angle_offset = Math.Round(RadiansToDegrees(pos.Angle),3);
                    
                    break;
                }
            }



            //FilteredElementCollector collector = new FilteredElementCollector(m_doc);
            //FilteredElementIterator iter = collector.OfCategory(BuiltInCategory.OST_Elev).GetElementIterator();
            //iter.Reset();
            //while (iter.MoveNext())
            //{
            //    Autodesk.Revit.DB.ElevationMarker elev = (ElevationMarker)iter.Current;
            //    Location loc = elev.Location; 
                
            //}

            FilteredElementCollector collector = new FilteredElementCollector(m_doc).OfClass(typeof(ViewSection));
            FilteredElementIterator iter = collector.GetElementIterator();
            while (iter.MoveNext())
            {
                ViewSection current = iter.Current as ViewSection;
                if (current == null) continue;

                ViewFamilyType vft = m_doc.GetElement(current.GetTypeId()) as ViewFamilyType;

                if (null == vft) continue;

                if (vft.Name == "Interior Elevation")
                    m_int_elevation_views.Add(current);
          
            }

            // Elev direction and location
            StringBuilder theValues = new StringBuilder("Rooms");
            Transaction tr = new Transaction(m_doc, "Rename Interior Elevations");
            tr.Start();

            foreach (Autodesk.Revit.DB.ViewSection v in m_int_elevation_views)
            {
                
                Debug.WriteLine(v.Name);
                // Construct a point at the midpoint of
                // the front bottom edge of the elev view's cropbox:
                XYZ pt = new XYZ(
                            v.CropBox.Max.X - ( (v.CropBox.Max.X - v.CropBox.Min.X) / 2),
                             1.0,
                             v.CropBox.Max.Z);

                // Get pt's translation to project's coordinate system:
                pt = v.CropBox.Transform.OfPoint(pt);


                
                
                // Get the room...
                Autodesk.Revit.DB.Architecture.Room rm = m_doc.GetRoomAtPoint(pt);
                if (null != rm)
                {
                    string viewDir = GetCardinalDirectionIndicator(ConvertViewDirectionToAngle(v.ViewDirection));
                    theValues.Append("\t" + v.Name + " -> " + rm.Name + " " + viewDir + "\n");
                    string newViewName = rm.Name + " " + viewDir;

                    
                    try
                    {
                        v.Name = newViewName;
                        
                    }
                    catch 
                    {
                        continue; // log error, or possibly duplicate
                    }
                    
                }
                else
                    theValues.Append("\t" + v.Name + " is not inside a room...\n");
                
            }

            tr.Commit();
            tr.Dispose(); // is this necessary?

            Debug.WriteLine(theValues.ToString());

        }

        // P R O P E R T I E S


        // M E T H O D S

        private double RadiansToDegrees(double radians)
        {
            return radians * (180 / Math.PI);
        }

        private double DegreesToRadians(double degrees)
        {
            return degrees * (Math.PI / 180);
        }

        private double NormalXY_ToRadians(double X, double Y)
        {
            return Math.Atan2(Y, X);
        }

        private double ConvertViewDirectionToAngle(XYZ ViewDirectionVector) // What about Project North direction?
        {
            // Get angle in radians:
            double rads = Math.Atan2(ViewDirectionVector.Y, ViewDirectionVector.X);
            // vector is toward elevation flag, so flip to get direction view is facing:
            if (rads < Math.PI)
                rads += Math.PI;
            else
                rads -= Math.PI;

            // Convert radians to degrees:
            return rads * (180 / Math.PI);
            
        }

        private string GetCardinalDirectionIndicator(double angle)
        {
            angle += m_north_angle_offset;
            string returnVal = "";
            if ((angle >= 0) && (angle < 45)) returnVal = "EAST";
            if ((angle >= 45) && (angle < 135)) returnVal = "NORTH";
            if ((angle >= 135) && (angle < 225)) returnVal = "WEST";
            if ((angle >= 225) && (angle < 315)) returnVal = "SOUTH";
            if ((angle >= 315) && (angle < 360)) returnVal = "EAST";
            
            return returnVal;

        }


    }
}
